<?php

class Rates_Calculator {

    // Step 1
    public $intNumberOfGolfers;
    public $intOccupancy;
    public $intDateStart;
    public $intDateEnd;
    public $arrRegions;

    // Step 2
    public $arrHotels;
    public $arrCourses;
    public $arrRatesHotels;
    public $arrRatesCourses;

    // Step 3
    public $arrModifiers;

    // Step 4
    public $fltGrandTotalHotel;
    public $fltGrandTotalHotelPrivate;
    public $fltGrandTotalGolf;
    public $fltGrandTotalGolfPrivate;
    public $fltGrandNetTotal;
    public $fltGrandNetTotalPrivate;
    public $fltBigTotal;
    public $fltBigTotalPrivate;
    public $fltNetMargin;
    public $fltNetMarginPrivate;
    public $fltCostPerPerson;
    public $fltCostPerPersonPrivate;

    public $fltGrandTotalFees;
    public $fltGrandTotalFeesPrivate;

    // Development variables
    public $blnCalculate;
    public $blnError;
    public $arrErrors;
    public $arrOutput;
    public $arrOutputPrivate;

    public function __construct($arrArgs = array()) {
        // Set the default args
        $arrArgs = Util::wp_parse_args($arrArgs, array(
            'calculate' => false,
            'number_of_golfers' => 2,
            'occupancy' => 2,
            'date_start' => date('Y-m-d'),
            'date_end' => date('Y-m-d'),
            'regions' => array(),
            'hotels' => array(),
            'courses' => array(),
            'rates_hotels' => array(),
            'rates_courses' => array(),
            'modifiers' => array(
                array(
                    'label' => 'GST',
                    'operation' => 'multiplication',
                    'value' => '0.05',
                ),
                array(
                    'label' => 'Hotel Tax',
                    'operation' => 'multiplication',
                    'value' => '0.08',
                ),
                array(
                    'label' => 'City Tax',
                    'operation' => 'multiplication',
                    'value' => '0.03',
                ),
                array(
                    'label' => 'Margin Profit',
                    'operation' => 'multiplication',
                    'value' => '0.15',
                ),
                array(
                    'label' => 'Margin Profit GST',
                    'operation' => 'multiplication',
                    'value' => '0.05',
                ),
                array(
                    'label' => 'CC Fees',
                    'operation' => 'multiplication',
                    'value' => '0.03',
                )
            )
        ));

        // Set step one variables
        $this->intNumberOfGolfers = intval($arrArgs['number_of_golfers']);
        $this->intOccupancy = intval($arrArgs['occupancy']);
        $this->intDateStart = strtotime($arrArgs['date_start']);
        $this->intDateEnd = strtotime($arrArgs['date_end']);
        $this->arrRegions = $arrArgs['regions'];

        // Set step two variables
        $this->arrHotels = (array) $arrArgs['hotels'];
        $this->arrRatesHotels = (array) $arrArgs['rates_hotels'];
        $this->arrCourses = (array) $arrArgs['courses'];
        $this->arrRatesCourses = (array) $arrArgs['rates_courses'];

        // Set step three variables
        $this->arrModifiers = (array) $arrArgs['modifiers'];

        // Set step four variables
        $this->fltGrandTotalHotel = 0;
        $this->fltGrandTotalHotelPrivate = 0;
        $this->fltGrandTotalGolf = 0;
        $this->fltGrandTotalGolfPrivate = 0;

        $this->fltGradTotalFees = 0;
        $this->fltGradTotalFeesPrivate = 0;

        $this->fltGrandNetTotal = 0;
        $this->fltGrandNetTotalPrivate = 0;
        $this->fltBigTotal = 0;
        $this->fltBigTotalPrivate = 0;
        $this->fltNetMargin = 0;
        $this->fltNetMarginPrivate = 0;
        $this->fltCostPerPerson = 0;
        $this->fltCostPerPersonPrivate = 0;

        // Set development variables
        $this->blnCalculate = $arrArgs['calculate'];
        $this->blnError = false;
        $this->arrErrors = array();
        $this->arrWarnings = array();
        $this->arrOutput = array();
        $this->arrOutputPrivate = array();
        $this->strPreamble = '';

        // Calculate the values if set
        if ($this->blnCalculate) {
            $this->calculate();
        }
    }

    public function calculate() {
        // validate values, need to check for 0 values
        $this->validateValues();

        if (!$this->blnError) {
            // Calculate the grand totals
            $this->calculateGrandTotalHotel();
            $this->calculateGrandTotalGolf();
            $this->calculateGrandNetTotal();

            // Calculate the big total
            $this->calculateBigTotal();

            // Calculate net margin
            $this->calculateNetMargin();

            // Calculate cost per person
            $this->calculateCostPerPerson();

            // Set the preamble text
            $this->formatPreamble();
        }
    }

    /**
     * Checks the calculator values to make sure they are valid
     *
     */
    public function validateValues() {

        // Required

        if (empty($this->intNumberOfGolfers)) {
            // Check the number of golfers
            $this->arrErrors[] = 'Please fill in the number of golfers.';
        }
        if (empty($this->intOccupancy)) {
            // Check the occupancy
            $this->arrErrors[] = 'Please fill in the occupancy amount.';
        }
        if (empty($this->intDateStart)) {
            // Check the start date
            $this->arrErrors[] = 'Invalid start date entered.';
        }
        if (empty($this->intDateEnd)) {
            // Check the end date
            $this->arrErrors[] = 'Invalid end date entered.';
        }
        if (empty($this->arrRegions)) {
            $this->arrErrors[] = 'Please select the regions.';
        }
        if (empty($this->arrModifiers)) {
            $this->arrErrors[] = 'No modifiers were entered.';
        }

        // Check the primary validation failed
        if (!empty($this->arrErrors)) {
            $this->blnError = true;
            return;
        }

        // Start of the secondary validation

        // Validate the dates
        if ($this->intDateStart > $this->intDateEnd) {
            $this->arrErrors[] = 'Date invalid, start date must not be after the end date.';
            $this->arrErrors[] = 'Date invalid, end date must not be before the start date.';
        }

        // Validate the modifiers
        foreach ($this->arrModifiers as $arrModifier) {
            if (!isset($arrModifier['operation']) || !isset($arrModifier['value'])) {
                $this->arrErrors = 'Modifier data is incomplete.';
            }
        }

        // Validate the hotels
        if (empty($this->arrHotels)) {
            // Check the hotels
            $this->arrWarnings[] = 'No hotels were selected.';
        } else {
            // Check the hotel rates
            if (empty($this->arrRatesHotels)) {
                $this->arrErrors[] = 'No hotel rates were selected.';
            } else {
                // Validate the hotel rates
                foreach ($this->arrRatesHotels as $arrRate) {
                    // Make sure all the data is there
                    if (!isset($arrRate['id']) || !isset($arrRate['nights']) || !isset($arrRate['rooms'])) {
                        $this->arrErrors[] = 'Hotel rate data is incomplete.';
                        continue;
                    }
                }
            }
        }

        // Validate the courses
        if (empty($this->arrCourses)) {
            // Check the hotels
            $this->arrWarnings[] = 'No courses were selected.';
        } else {
            // Check the course rates
            if (empty($this->arrRatesCourses)) {
                $this->arrErrors[] = 'No course rates were selected.';
            } else {
                // Validate the course rates
                foreach ($this->arrRatesCourses as $arrRate) {
                    // Make sure all the data is there
                    if (!isset($arrRate['id']) || !isset($arrRate['rounds'])) {
                        $this->arrErrors[] = 'Course rate data is incomplete.';
                        continue;
                    }
                }
            }
        }

        // Check the secondary validation
        if (!empty($this->arrErrors)) {
            $this->blnError = true;
        }
    }

    /**
     * Calculates the totals for the hotels
     *
     */
    public function calculateGrandTotalHotel() {
        if (!empty($this->arrRatesHotels)) {

            $arrTotalFinal = array();
            $arrTotalFinalPrivate = array();
            $this->arrOutput[] = 'Grand Total Hotel:';
            $this->arrOutputPrivate[] = 'Grade Total Hotel (Private rates):';

            foreach ($this->arrRatesHotels as $i => $arrRateHotel) {

                // Set hotel variables
                $intId = (int) $arrRateHotel['id'];
                $intNights = !empty($arrRateHotel['nights']) ? (int) $arrRateHotel['nights'] : 1;
                $intRooms = !empty($arrRateHotel['rooms']) ? (int) $arrRateHotel['rooms'] : 1;

                // Get the rate
                $arrRate = Rate::getHotelRateById($intId);
                $strPrice = !empty($arrRate['price']) ? $arrRate['price'] : 0;
                $strPrivatePrice = !empty($arrRate['private_price']) ? $arrRate['private_price'] : $strPrice;

                $strName = !empty($arrRate['name']) ? $arrRate['name'] : '';

                // If rate price is not set
                if (empty($strPrice)) {
                    $this->blnError = true;
                    $this->arrErrors = 'Hotel rate <strong>' . $strName . '</strong> price is not set.';
                    continue;
                }

                // Calculate the total before tax
                $fltBaseCost = $intNights * $intRooms * $strPrice;
                $fltBasePrivateCost = $intNights * $intRooms * $strPrivatePrice;

                $arrTotal = array(
                    $fltBaseCost
                );

                $arrTotalPrivate = array(
                    $fltBasePrivateCost
                );

                // Add to output
                $this->arrOutput[] = $intNights . ' * ' . $intRooms . ' * ' . $strPrice . ' = ' . $fltBaseCost;
                $this->arrOutputPrivate[] = $intNights . ' * ' . $intRooms . ' * ' . $strPrivatePrice . ' = ' . $fltBasePrivateCost;

                // Add first three modifiers
                for ($i=0; $i<3; $i++) {
                    if (isset($this->arrModifiers[$i])) {
                        // Calculate the modifier
                        $fltValue = static::calculateModifier($this->arrModifiers[$i], $fltBaseCost);
                        $fltPrivateValue = static::calculateModifier($this->arrModifiers[$i], $fltBasePrivateCost);
                        $arrTotal[] = $fltValue;
                        $arrTotalPrivate[]  = $fltPrivateValue;

                        // Add to output
                        $this->arrOutput[] = $fltBaseCost . ' * ' . $this->arrModifiers[$i]['value'] . ' = ' . $fltValue;
                        $this->arrOutputPrivate[] = $fltBasePrivateCost . ' * ' . $this->arrModifiers[$i]['value'] . ' = ' . $fltPrivateValue;
                    }
                }

                if (count($arrRate['fees']) > 0) {
                    foreach ($arrRate['fees'] As $arrRateFee) {
                        if ($arrRateFee['calculation_type'] == 'flat') {
                            $fltFeeValue = $arrRateFee['amount'];
                        } else {
                            $fltFeeValue = $fltBaseCost * ($arrRateFee['amount'] / 100);
                        }

                        if ($arrRateFee['public'] == '1') {
                            $arrTotal[] = $fltFeeValue;
                            $this->arrOutput[] = " + " . $fltFeeValue . " (" . $arrRateFee['name'] . ")";
                        }

                        if ($arrRateFee['private'] == '1') {
                            $arrTotalPrivate[] = $fltFeeValue;
                            $this->arrOutputPrivate[] = " + " . $fltFeeValue . " (" . $arrRateFee['name'] . ")";
                        }
                    }
                }

                // Add the hotel values
                $fltTotal = array_sum($arrTotal);
                $arrTotalFinal[] = $fltTotal;

                // Add the hotel values (private rates)
                $fltTotalPrivate = array_sum($arrTotalPrivate);
                $arrTotalFinalPrivate[] = $fltTotalPrivate;

                // Add to output
                $this->arrOutput[] = implode(' + ', $arrTotal) . ' = ' . $fltTotal;
                $this->arrOutput[] = '---';

                // Add to output (Private Rates)
                $this->arrOutputPrivate[] = implode(' + ', $arrTotalPrivate) . ' = ' . $fltTotalPrivate;
                $this->arrOutputPrivate[] = '---';
            }

            // Add all the totals together
            $fltTotalFinal = array_sum($arrTotalFinal);
            $this->fltGrandTotalHotel = $fltTotalFinal;

            // Add all the totals together (private rates)
            $fltTotalFinalPrivate = array_sum($arrTotalFinalPrivate);
            $this->fltGrandTotalHotelPrivate = $fltTotalFinalPrivate;

            // Add to output
            $this->arrOutput[] = implode(' + ', $arrTotalFinal) . ' = ' . $fltTotalFinal;
            $this->arrOutput[] = '---';

            // Add to output (private rates)
            $this->arrOutputPrivate[] = implode(' + ', $arrTotalFinalPrivate) . ' = ' . $fltTotalFinalPrivate;
            $this->arrOutputPrivate[] = '---';
        }
    }

    /**
     * Calculates the totals for the golf courses
     *
     */
    public function calculateGrandTotalGolf() {
        if (!empty($this->arrCourses)) {

            // Array of all course prices
            $arrTotalFinal = array();
            $arrTotalFinalPrivate = array();
            $this->arrOutput[] = 'Grand Total Golf:';
            $this->arrOutputPrivate[] = 'Grand Total Golf (Private):';

            foreach ($this->arrRatesCourses as $i => $arrRateCourse) {
                // Set the course variables
                $intId = (int) $arrRateCourse['id'];
                $intRounds = !empty($arrRateCourse['rounds']) ? (int) $arrRateCourse['rounds'] : 1;

                // Get the rate
                $arrRate = Rate::getCourseRateById($intId);
                $strPrice = !empty($arrRate['price']) ? $arrRate['price'] : 0;
                $strPricePrivate = !empty($arrRate['private_price']) ? $arrRate['private_price'] : $strPrice;
                $strName = !empty($arrRate['name']) ? $arrRate['name'] : 0;

                // If rate price is not set
                if (empty($strPrice)) {
                    $this->blnError = true;
                    $this->arrErrors = 'Golf course rate <strong>' . $strName . '</strong> price is not set.';
                    continue;
                }

                // Calculate the total before tax
                $fltBaseCost = $this->intNumberOfGolfers * $intRounds * $strPrice;
                $fltBaseCostPrivate = $this->intNumberOfGolfers * $intRounds * $strPricePrivate;

                $arrTotal = array(
                    $fltBaseCost
                );

                $arrTotalPrivate = array(
                    $fltBaseCostPrivate
                );

                // Add to output
                $this->arrOutput[] = $this->intNumberOfGolfers . ' * ' . $intRounds . ' * ' . $strPrice . ' = ' . $fltBaseCost;
                $this->arrOutputPrivate[] = $this->intNumberOfGolfers . ' * ' . $intRounds . ' * ' . $strPricePrivate . ' = ' . $fltBaseCostPrivate;

                // Add first modifier (GST)
                for ($i=0; $i<1; $i++) {
                    if (isset($this->arrModifiers[$i])) {
                        // Calculate the modifier
                        $fltValue = static::calculateModifier($this->arrModifiers[$i], $fltBaseCost);
                        $fltValuePrivate = static::calculateModifier($this->arrModifiers[$i], $fltBaseCostPrivate);
                        $arrTotal[] = $fltValue;
                        $arrTotalPrivate[] = $fltValuePrivate;

                        // Add to output
                        $this->arrOutput[] = $fltBaseCost . ' * ' . $this->arrModifiers[$i]['value'] . ' = ' . $fltValue;
                        $this->arrOutputPrivate[] = $fltBaseCost . ' * ' . $this->arrModifiers[$i]['value'] . ' = ' . $fltValuePrivate;
                    }
                }

                if (count($arrRate['fees']) > 0) {
                    foreach ($arrRate['fees'] As $arrRateFee) {
                        if ($arrRateFee['calculation_type'] == 'flat') {
                            $fltFeeValue = $arrRateFee['amount'];
                        } else {
                            $fltFeeValue = $fltBaseCost * ($arrRateFee['amount'] / 100);
                        }

                        if ($arrRateFee['public'] == '1') {
                            $arrTotal[] = $fltFeeValue;
                            $this->arrOutput[] = " + " . $fltFeeValue . " (" . $arrRateFee['name'] . ")";
                        }

                        if ($arrRateFee['private'] == '1') {
                            $arrTotalPrivate[] = $fltFeeValue;
                            $this->arrOutputPrivate[] = " + " . $fltFeeValue . " (" . $arrRateFee['name'] . ")";
                        }
                    }
                }

                // Add the course total
                $fltTotal = array_sum($arrTotal);
                $arrTotalFinal[] = $fltTotal;

                $fltTotalPrivate = array_sum($arrTotalPrivate);
                $arrTotalFinalPrivate[] = $fltTotalPrivate;

                // Add to output
                $this->arrOutput[] = implode(' + ', $arrTotal) . ' = ' . $fltTotal;
                $this->arrOutput[] = '---';
                $this->arrOutputPrivate[] = implode(' + ', $arrTotalPrivate) . ' = ' . $fltTotalPrivate;
                $this->arrOutputPrivate[] = '---';
            }

            // Add all the totals together
            $fltTotalFinal = array_sum($arrTotalFinal);
            $fltTotalFinalPrivate = array_sum($arrTotalFinalPrivate);
            $this->fltGrandTotalGolf = $fltTotalFinal;
            $this->fltGrandTotalGolfPrivate = $fltTotalFinalPrivate;

            // Add to output
            $this->arrOutput[] = implode(' + ', $arrTotalFinal) . ' = ' . $fltTotalFinal;
            $this->arrOutput[] = '---';

            $this->arrOutputPrivate[] = implode(' + ', $arrTotalFinalPrivate) . ' = ' . $fltTotalFinalPrivate;
            $this->arrOutputPrivate[] = '---';
        }
    }

    /**
     * Calculates the Grands Net Total
     *
     */
    public function calculateGrandNetTotal() {
        // Add the two grand totals together
        $fltTotal = $this->fltGrandTotalHotel + $this->fltGrandTotalGolf;
        $this->fltGrandNetTotal = $fltTotal;

        $fltTotalPrivate = $this->fltGrandTotalHotelPrivate + $this->fltGrandTotalGolfPrivate;
        $this->fltGrandNetTotalPrivate = $fltTotalPrivate;

        // Add to output
        $this->arrOutput[] = 'Grand Net Total:';
        $this->arrOutput[] = $this->fltGrandTotalHotel . ' + ' . $this->fltGrandTotalGolf . ' = ' . $fltTotal;
        $this->arrOutput[] = '---';

        $this->arrOutputPrivate[] = 'Grand Net Total (Private):';
        $this->arrOutputPrivate[] = $this->fltGrandTotalHotelPrivate . ' + ' . $this->fltGrandTotalGolfPrivate . ' = ' . $fltTotalPrivate;
        $this->arrOutputPrivate[] = '---';
    }

    /**
     * Calculates the Big Total
     *
     */
    public function calculateBigTotal() {
        // Make an array with all the totals
        $fltTotal = $this->fltGrandNetTotal;
        $arrTotals = array(
            $fltTotal
        );

        $fltTotalPrivate = $this->fltGrandNetTotalPrivate;
        $arrTotalsPrivate = array(
            $fltTotalPrivate
        );

        // Add to output
        //$this->arrOutput[] = 'Big Total:';
        $this->arrOutputPrivate[] = 'Big Total (Private):';


        if (isset($this->arrModifiers[3]) && isset($this->arrModifiers[4])) {
            // Calculate the margin and margin GST
            //$fltMargin = static::calculateModifier($this->arrModifiers[3], $fltTotal);
            //$fltMarginGst = static::calculateModifier($this->arrModifiers[4], $fltMargin);
            //$arrTotals[] = $fltMargin;
            //$arrTotals[] = $fltMarginGst;

            $fltMarginPrivate = static::calculateModifier($this->arrModifiers[3], $fltTotalPrivate);
            $fltMarginGstPrivate = static::calculateModifier($this->arrModifiers[4], $fltMarginPrivate);
            $arrTotalsPrivate[] = $fltMarginPrivate;
            $arrTotalsPrivate[] = $fltMarginGstPrivate;

            // Add to output
            //$this->arrOutput[] = $fltTotal . ' * ' . $this->arrModifiers[3]['value'] . ' = ' . $fltMargin;
            //$this->arrOutput[] = $fltMargin . ' * ' . $this->arrModifiers[4]['value'] . ' = ' . $fltMarginGst;

            $this->arrOutputPrivate[] = $fltTotalPrivate . ' * ' . $this->arrModifiers[3]['value'] . ' = ' . $fltMarginPrivate;
            $this->arrOutputPrivate[] = $fltMarginPrivate . ' * ' . $this->arrModifiers[4]['value'] . ' = ' . $fltMarginGstPrivate;
        }

        // Add the margin totals
        //$fltMarginTotal = array_sum($arrTotals);
        $fltMarginTotalPrivate = array_sum($arrTotalsPrivate);

        // Add to output
        //$this->arrOutput[] = implode(' + ', $arrTotals) . ' = ' . $fltMarginTotal;
        $this->arrOutputPrivate[] = implode(' + ', $arrTotalsPrivate) . ' = ' . $fltMarginTotalPrivate;

        if (isset($this->arrModifiers[5])) {
            // Calculate the CC Fees
            //$fltCcFees = static::calculateModifier($this->arrModifiers[5], $fltMarginTotal);
            //$arrTotals[] = $fltCcFees;

            $fltCcFeesPrivate = static::calculateModifier($this->arrModifiers[5], $fltMarginTotalPrivate);
            $arrTotalsPrivate[] = $fltCcFeesPrivate;

            // Add to output
            //$this->arrOutput[] = $fltMarginTotal . ' * ' . $this->arrModifiers[5]['value'] . ' = ' . $fltCcFees;
            $this->arrOutputPrivate[] = $fltMarginTotalPrivate . ' * ' . $this->arrModifiers[5]['value'] . ' = ' . $fltCcFeesPrivate;
        }

        // Add all the totals together
        $fltTotal = array_sum($arrTotals);
        $this->fltBigTotal = $fltTotal;

        $fltTotalPrivate = array_sum($arrTotalsPrivate);
        $this->fltBigTotalPrivate = $fltTotalPrivate;

        // Add to output
        //$this->arrOutput[] = implode(' + ', $arrTotals) . ' = ' . $fltTotal;
        //$this->arrOutput[] = '---';

        $this->arrOutputPrivate[] = implode(' + ', $arrTotalsPrivate) . ' = ' . $fltTotalPrivate;
        $this->arrOutputPrivate[] = '---';
    }

    /**
     * Calculates the net margins
     *
     */
    public function calculateNetMargin() {
        if (isset($this->arrModifiers[3])) {

            // Apply 4th modifier (margin profit) to the grand net total
            //$fltValue = static::calculateModifier($this->arrModifiers[3], $this->fltGrandNetTotal);
            //$this->fltNetMargin = $fltValue;

            $fltValuePrivate = static::calculateModifier($this->arrModifiers[3], $this->fltGrandNetTotalPrivate);
            $this->fltNetMarginPrivate = $fltValuePrivate;

            // Add to output
            //$this->arrOutput[] = 'Net Margin:';
            //$this->arrOutput[] = $this->fltGrandNetTotal . ' * ' . $this->arrModifiers[3]['value'] . ' = ' . $fltValue;
            //$this->arrOutput[] = '---';

            $this->arrOutputPrivate[] = 'Net Margin:';
            $this->arrOutputPrivate[] = $this->fltGrandNetTotalPrivate . ' * ' . $this->arrModifiers[3]['value'] . ' = ' . $fltValuePrivate;
            $this->arrOutputPrivate[] = '---';

        }
    }

    /**
     * Calculates the cost per person
     *
     */
    public function calculateCostPerPerson() {
        // Divide the big total by the occupancy
        $fltValue = $this->fltBigTotal / $this->intOccupancy;
        $this->fltCostPerPerson = $fltValue;

        $fltValuePrivate = $this->fltBigTotalPrivate / $this->intOccupancy;
        $this->fltCostPerPersonPrivate = $fltValuePrivate;

        // Add to output
        $this->arrOutput[] = 'Cost Per Person:';
        $this->arrOutput[] = $this->fltBigTotal . ' / ' . $this->intOccupancy . ' = ' . $fltValue;
        $this->arrOutput[] = '---';

        $this->arrOutputPrivate[] = 'Cost Per Person:';
        $this->arrOutputPrivate[] = $this->fltBigTotalPrivate . ' / ' . $this->intOccupancy . ' = ' . $fltValuePrivate;
        $this->arrOutputPrivate[] = '---';
    }

    /**
     * Returns the calculator totals
     *
     */
    public function getTotals() {
        // Make an array with all our totals
        $arrTotals = array();

        $arrTotals['public'] = array(
            'Grand Total Hotel' => $this->fltGrandTotalHotel,
            'Grand Total Golf' => $this->fltGrandTotalGolf,
            'Grand Net Total' => $this->fltGrandNetTotal,
            'Big Total' => $this->fltBigTotal,
            'Net Margin' => $this->fltNetMargin,
            'Cost Per Person' => $this->fltCostPerPerson
        );

        $arrTotals['private'] = array(
            'Grand Total Hotel' => $this->fltGrandTotalHotelPrivate,
            'Grand Total Golf' => $this->fltGrandTotalGolfPrivate,//$this->fltGrandTotalGolf,
            'Grand Net Total' => $this->fltGrandNetTotalPrivate,//$this->fltGrandNetTotal,
            'Big Total' => $this->fltBigTotalPrivate,//$this->fltBigTotal,
            'Net Margin' => $this->fltNetMarginPrivate,//$this->fltNetMargin,
            'Cost Per Person' => $this->fltCostPerPersonPrivate //$this->fltCostPerPerson
        );

        // Return them
        return $arrTotals;
    }

    /**
     * Format the preamble
     *
     */
    public function formatPreamble() {
        $objDate = new DateTime();
        $strDate = $objDate->format('F d, Y');


        // Set the format
        $strFormat = $strDate . "\n\n"
                   . 'Trip Quote for {NAME} for %d golfers based on %s sharing %d rooms.' . "\n\n"
                   . 'From %s to %s (%d nights) travelling to the %s region.' . "\n\n"
                   . 'Golfing at:' . "\n" . '%s' . "\n"
                   //. "{GROUP MAKEUP COMMENTS} \n\n"
                   . 'and staying at:' . "\n" . '%s' . "\n"
                   . 'Your complete package price includes shared power carts, pre-booked tee times, range use where noted** as well as all taxes and resort fees (if applicable). Meals, transportation and parking are not included, unless noted.' . "\n\n"
                   . '$ per person based on double / quad occupancy' . "\n\n"
                   . 'Your quote is valid for {DAYS} days.';

        // Set the data
        $arrData = array(
            'golfers'    => $this->intNumberOfGolfers,
            'occupancy'  => $this->intOccupancy,
            'rooms'      => $this->countRooms(),
            'start_date' => date('F d, Y', $this->intDateStart),
            'end_date'   => date('F d, Y', $this->intDateEnd),
            'nights'     => $this->countNights(),
            'regions'    => $this->formatRegionList(),
            'courses'    => $this->formatCourseList(),
            'hotels'     => $this->formatHotelList(),
        );

        // Set the template
        $this->strPreamble = vsprintf($strFormat, $arrData);
    }

    /**
     * Gets the number of rooms based on the hotel rates selected
     *
     */
    public function countRooms() {
        $intRooms = 0;
        foreach ($this->arrRatesHotels as $arrRate) {
            $intRooms += (int) $arrRate['rooms'];
        }

        return $intRooms;
    }

    /**
     * Gets the number of nights based on the hotel rates selected
     *
     */
    public function countNights() {
        $intNights = 0;
        foreach ($this->arrRatesHotels as $arrRate) {
            $intNights += (int) $arrRate['nights'];
        }

        return $intNights;
    }

    /**
     * Formats the list of regions for the preamble
     *
     */
    public function formatRegionList() {
        // Set the region object
        $objRegion = new Region;

        $arrList = array();
        foreach ($this->arrRegions as $intId) {
            $arrRegion = $objRegion->getDetail($intId);
            $arrList[] = $arrRegion['region_name'];
        }

        return implode(', ', $arrList);
    }

    /**
     * Formats the list of courses for the preamble
     *
     */
    public function formatCourseList() {
        $strList = '';
        foreach ($this->arrCourses as $intId) {
            $arrCourse = Rate::getCourseById($intId);
            $strList .= sprintf(" - %s\n", $arrCourse['course_name']);
        }

        return $strList;
    }

    /**
     * Formats the list of hotels for the preamble
     *
     */
    public function formatHotelList() {
        $strList = '';
        foreach ($this->arrRatesHotels as $arrRateSelected) {
            $arrRate = Rate::getHotelRateById($arrRateSelected['id']);
            $arrHotel = Rate::getHotelById($arrRate['hotel_id']);
            $strList .= sprintf(" - %s (%s)\n", $arrHotel['hotel_name'], $arrRate['name']);
        }

        return $strList;
    }

    /**
     * Outputs the messages from the calculator
     *
     */
    public function displayMessages() {
        // Start the output
        $strOutput = '';

        // Output the errors
        if (!empty($this->arrErrors)) {
            $strOutput .= '<div class="alert alert-danger">' . implode('<br>', $this->arrErrors) . '</div>';
        }

        // Output the warnings
        if (!empty($this->arrWarnings)) {
            $strOutput .= '<div class="alert alert-warning">' . implode('<br>', $this->arrWarnings) . '</div>';
        }

        echo $strOutput;
    }

    /**
     * Displays the output for customers
     *
     */
    public function displayOutput() {
        // Display the output
        if (!empty($this->arrOutput)) {
            echo '<code>' . implode('<br>', $this->arrOutput) . '</code>';
        }
    }

    /**
     * Displays the private output used for internal margin calculations
     *
     */
    public function displayOutputPrivate() {
        // Display the output
        if (!empty($this->arrOutputPrivate)) {
            echo '<code>' . implode('<br>', $this->arrOutputPrivate) . '</code>';
        }
    }

    /**
     * Displays the formatted preamble text
     *
     */
    public function displayPreamble() {
        // Display the output
        if (!empty($this->strPreamble)) {
            echo '<textarea class="form-control" rows="10">' . $this->strPreamble . '</textarea>';
        }
    }

    /**
     * Calculates the value of a modifier
     *
     */
    public static function calculateModifier($arrModifier, $fltTotal = 0) {
        // Return the original total if value is empty
        if (empty($arrModifier['value'])) {
            return 0;
        }

        // Make sure the total has a value for multiply and divide
        if (!empty($fltTotal)) {
            if ($arrModifier['operation'] == 'multiplication') {
                // Multiply
                return $fltTotal * $arrModifier['value'];
            }
            if ($arrModifier['operation'] == 'division') {
                // Divide
                return $fltTotal / $arrModifier['value'];
            }
        }

        if ($arrModifier['operation'] == 'addition') {
            // Add
            return $fltTotal + $arrModifier['value'];
        }

        if ($arrModifier['operation'] == 'subtraction') {
            // Subtract
            return $fltTotal - $arrModifier['value'];
        }

        // fallback
        return 0;
    }


}
