/*************
* Parent Type Changer Control
* By: Liam Anderson
* Company: Navigator Multimedia
* Date: 2015-10-15
*
* Descriptin: Control that allows user to select a type (hotel, course) by searching.  Uses AJAX to load results
*
* Usage: Add an element with the class of "js-ctrl-type-changer" and use the data params below
* 
* Options: (via data attributes)
*	data-type: required (or course) - string: either "hotel" or "course"
*	data-input-id: optional - string: id of the input values will be saved too (type ID is loaded into)
*	data-input-class: optional - string: class for the input values will be saved too (type ID is loaded into)
*	data-input-name: optional - string: name for the input values will be saved too (type ID is loaded into)
*	data-default-id: optional - integer: id of the type to be preloaded
*	data-ignore-id: optional - integer: id of type to now allow user to select (usually the current type we are changing the parent for)
*	data-event-change: optional - function reference: a callback function for after the value has been changed (similar to onclick="<js code>");
*	data-instance-id: optional - string: a unique string to identify this instance of the control, will be auto genetyped if not passed in\
*
* Example: <div class="js-ctrl-type-changer" data-type="hotel" data-input-id="my-save-input" data-input-name="parent_id" data-default-id="5" data-event-change="functionReference"><?php // drivin and rendered in javascript ?></div>
*
**************/
var objTypeChangers = {};

(function($){

	// control class
	var Type_Changer = function(divElement){
		var objThis = this;
		objThis.element = $(divElement);
		objThis.type = null;
		objThis.defaultId = null;
		objThis.ignoreId = null;
		objThis.inputId = null;
		objThis.inputClass= null;
		objThis.inputName = null;
		objThis.events = {};
		objThis.instanceId = null;
		objThis.modal = null;
		objThis.types = {};
		objThis.selectedType = {
			id: null,
			name: null
		};

		// pull in values from data attributes
		objThis.initializeValues(function(){
			objThis.createBaseHTML()
				.setupEvents()
				.search('');

			// save instance to list for access outside of this file
			objTypeChangers[objThis.instanceId] = objThis;
		});
	}

	Type_Changer.prototype.initializeValues = function(fnCallback){
		var objThis = this;

		objThis.instanceId = guid(); // unique instance id used for this control

		var strInstanceId = objThis.element.data('instance-id');
		if(objThis.element.attr('data-instance-id')){
			objThis.instanceId = strInstanceId;
		}

		var strType = objThis.element.data('type');
		if(objThis.element.attr('data-type')){
			objThis.type = strType;
		}

		var strDefaultId = objThis.element.data('default-id');
		if(objThis.element.attr('data-default-id')){
			objThis.defaultId = parseInt(strDefaultId);
		}

		var strIgnoreId = objThis.element.data('ignore-id');
		if(objThis.element.attr('data-ignore-id')){
			objThis.ignoreId = parseInt(strIgnoreId);
		}

		objThis.inputId = 'tc-value-input-' + objThis.instanceId;
		var strInputId = objThis.element.data('input-id');
		if(objThis.element.attr('data-input-id')){
			objThis.inputId = strInputId;
		}

		var strInputClass = objThis.element.data('input-class');
		if(objThis.element.attr('data-input-class')){
			objThis.inputClass = strInputClass;
		}

		var strInputName = objThis.element.data('input-name');
		if(objThis.element.attr('data-input-name')){
			objThis.inputName = strInputName;
		}

		var cbEventChanged = objThis.element.data('event-change');
		if(objThis.element.attr('data-event-change')){
			objThis.events['change'] = cbEventChanged;
		}

		if(objThis.defaultId != null){
			// get default values if one is passed in
			objThis.getTypeById(objThis.type, objThis.defaultId, fnCallback);
			return objThis;
		}

		// continue without loading a default
		if(typeof(fnCallback) == 'function'){
			fnCallback();
		}

		return objThis;
	}

	Type_Changer.prototype.createBaseHTML = function(){
		var objThis = this;

		var strInputName = '';
		if(objThis.inputName != null){
			strInputName = 'name="' + objThis.inputName + '"';
		}

		var strInputClass = '';
		if(objThis.inputClass != null){
			strInputClass = objThis.inputClass;
		}

		var strSelectedName = 'No ' + objThis.type.capitalizeFirstLetter() + ' Selected';
		var strSelectedId = '';

		if(objThis.selectedType.id != null){
			strSelectedId = objThis.selectedType.id;
			strSelectedName = objThis.selectedType.name;
		}

		var strHTML = '<input type="hidden" id="' + objThis.inputId + '" ' + strInputName + ' class="js-tc-value-input js-tc-value-input-' + objThis.instanceId + ' ' + strInputClass + '" value="' + strSelectedId + '" />'
					+ '<span class="js-tc-value-display">' + strSelectedName + '</span><a href="#" class="btn btn-link js-tc-open-modal">Change</a>'

		// attach to element
		objThis.element.html('').append(strHTML);

		var strModalHTML = '<div class="js-tc-holder">'
				    	+ 	'<div class="modal fade" id="tc-modal-' + objThis.instanceId + '" tabindex="-1" role="dialog" aria-labelledby="tc-modal-' + objThis.instanceId + '-label" style="z-index: 5000;">'
				        + 	'<div class="modal-dialog" role="document">'
				        + 	'<div class="modal-content">'
				        + 	'<div class="modal-header">'
				        + 	'<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>'
				        + 	'<h4 class="modal-title" id="tc-modal-' + objThis.instanceId + '-label">Select ' + objThis.type.capitalizeFirstLetter() + '</h4>'
				        + 	'</div>'
				        + 	'<div class="modal-body">'
				        + 	'<input type="text" class="form-control js-tc-search" placeholder="Search" />'
						+ 	'<table class="js-tc-results table table-hover">'
				        + 	'</table>'
				        + 	'</div>'
				        + 	'<div class="modal-footer">'
				        + 	'<button type="button" class="btn btn-default" data-dismiss="modal">Close</button>'
				        + 	'</div>'
				        + 	'</div>'
				        + 	'</div>'
				   		+ 	'</div>'
						+ 	'</div>';

		// attach modal
		if(objThis.modal != null){
			// remove existing modal
			objThis.modal.parent().remove();
		}

		$('body').append(strModalHTML);
		objThis.modal = $('.js-tc-holder #tc-modal-' + objThis.instanceId);

		objThis.modal.on('shown.bs.modal', function(event){
			// move keyboard focus to search input
			$(this).find('.js-tc-search').focus();

			return true;
		});

		return objThis;
	}

	Type_Changer.prototype.setupEvents = function(){
		var objThis = this;

		var jobjModal = $('#tc-modal-' + objThis.instanceId);

		objThis.element.on('click', '.js-tc-open-modal', function(event){
			event.preventDefault();

			objThis.showModal();
		});

		jobjModal.find('.js-tc-search').on('keyup', function(event){
			objThis.search($(this).val());
		});

		jobjModal.find('.js-tc-results').on('click', '.js-tc-select-type', function(event){
			var intTypeId = $(this).data('type-id');

			objThis.selectedType.id = intTypeId;

			if(intTypeId != null){
				objThis.selectedType.name = objThis.types[intTypeId].name;
			}else{
				objThis.selectedType.name = null;
			}

			objThis.setSelectedType().hideModal();
		});

		return objThis;
	}

	Type_Changer.prototype.search = function(strSearch){
		var objThis = this;

		var strSearchType = 'base';
		if(strSearch.length > 0){
			// search all types based on strSearch
			strSearchType = 'search';
		}

		$.getJSON('/admin/rates/' + objThis.type + '/searchtype/' + encodeURIComponent(strSearch), {}, function(objReturn){
			if(objReturn.error === true){
				objThis.modal.find('.js-tc-results').html('<tr><td>No ' + objThis.type.capitalizeFirstLetter() + ' Found.</td></tr>');
				objThis.types = null;

				// error occured
				console.log('Unknown Error Occured: ' + objReturn.messages);
				alert('Unknown Error Occured: ' + objReturn.messages.toString());
				return false;
			}

			// success
			$.extend(objThis.types, objReturn.types);

			objThis.createTableRows(objReturn.types);

			return true;
		});

		return objThis;
	}

	Type_Changer.prototype.setSelectedType = function(){
		var objThis = this;

		var strSelectedName = 'No ' + objThis.type.capitalizeFirstLetter() + ' Selected';
		var strSelectedId = '';

		if(objThis.selectedType.id != null){
			strSelectedId = objThis.selectedType.id;
			strSelectedName = objThis.selectedType.name;
		}

		// save type id to input
		objThis.element.find('.js-tc-value-input-' + objThis.instanceId).val(strSelectedId);

		// change label
		objThis.element.find('.js-tc-value-display').html(strSelectedName);

		if(typeof(objThis.events['change']) == 'function'){
			objThis.events['change'].call();
		}else{
			var fnCallback = eval(objThis.events['change']);

			if(typeof(objThis.events['change']) == 'function'){
				objThis.events['change'].call();
			}
		}

		return objThis;
	}

	Type_Changer.prototype.getSelectedType = function(){
		var objThis = this;

		return objThis.selectedType;
	}

	Type_Changer.prototype.showModal = function(){
		var objThis = this;

		objThis.modal.modal('show');

		return objThis;
	}

	Type_Changer.prototype.hideModal = function(){
		var objThis = this;

		objThis.modal.modal('hide');

		return objThis;
	}

	Type_Changer.prototype.getTypeById = function(intType, intTypeId, fnCallback){
		var objThis = this;

		// make ajax call to get default type information
		$.getJSON('/admin/rates/' + objThis.type + '/gettype/' + intTypeId, {}, function(objReturn){
			if(objReturn.error === true){
				// error occured
				console.log('Unknown Error Occured: ' + objReturn.messages);
				alert('Unknown Error Occured: ' + objReturn.messages.toString());
				return false;
			}

			// success
			objThis.types[objReturn.type.id] = objReturn.type;
			
			objThis.selectedType.id = objReturn.type.id;

			if(objReturn.type.name != null && objReturn.type.name.length > 0){
				objThis.selectedType.name = objReturn.type.name;
			}else{
				objThis.selectedType.name = '#' + objReturn.type.id + ' Type';
			}

			if(typeof(fnCallback) == 'function'){
				fnCallback();
				return objThis;
			}
		});

		return objThis;
	}

	Type_Changer.prototype.createTableRows = function(objTypes){
		var objThis = this;
		var jobjTable = objThis.modal.find('.js-tc-results');

		if(isEmptyObject(objTypes)){
			// no types found
			jobjTable.html('<tr><td>No Types Found.</td></tr>');
			return objThis;
		}

		var strHTML = '';

		// setup null row
		strHTML += objThis.createTableRow({
			id: null,
			name: 'No ' + objThis.type.capitalizeFirstLetter()
		});

		for(var intTypeId in objTypes){
			strHTML += objThis.createTableRow(objTypes[intTypeId]);
		}

		jobjTable.html(strHTML);

		return objThis;
	}

	Type_Changer.prototype.createTableRow = function(objType){
		var objThis = this;

		var strId = '';
		if(objType.id != null){
			strId = objType.id;
		}

		if(objType.name == null || objType.name.length == 0){
			// default name if there isn't one set
			objType.name = '#' + objType.id + ' ' + objThis.type.capitalizeFirstLetter();
		}

		strHTML = '<tr class="main-type" data-type-id="' + objType.id + '">'
				+ '<td>' + strId + '</td>'
				+ '<td>' + objType.name + '</td>';

		if(objThis.ignoreId == null || objThis.ignoreId != objType.id){
			strHTML += '<td><button class="btn btn-primary js-tc-select-type" data-type-id="' + objType.id + '">select</button></td>';
		}else{
			strHTML += '<td>&nbsp;</td>';
		}

		strHTML += '</tr>';

		return strHTML;
	}


	// setup controls
	$(document).ready(function(){
		$('.js-ctrl-type-changer').each(function(){
			var objThis = this;

			var objTypeChanger = new Type_Changer(objThis);
		});
	});

})(jQuery)