/*************
* Parent Rate Changer Control
* By: Liam Anderson
* Company: Navigator Multimedia
* Date: 2015-10-15
*
* Descriptin: Control that allows user to select a parent ID (or really a rate) with a tree structure and searching.  Uses AJAX to load results
*
* Usage: Add an element with the class of "js-ctrl-parent-rate-changer" and use the data params below
* 
* Options: (via data attributes)
*	data-hotel-id: required (or course) - integer: hotel for the rates to lookup
*	data-course-id: required (or hotel) - integer: course for the rates to lookup
*	data-input-id: optional - string: id of the input values will be saved too (rate ID is loaded into)
*	data-input-name: optional - string: name for the input values will be saved too (rate ID is loaded into)
*	data-default-id: optional - integer: id of the rate to be preloaded
*	data-ignore-id: optional - integer: id of rate to now allow user to select (usually the current rate we are changing the parent for)
*	data-event-change: optional - function reference: a callback function for after the value has been changed (similar to onclick="<js code>");
*	data-instance-id: optional - string: a unique string to identify this instance of the control, will be auto generated if not passed in\
*
* Example: <div class="js-ctrl-parent-rate-changer" data-hotel-id="1" data-input-id="my-save-input" data-input-name="parent_id" data-default-id="5" data-event-change="functionReference"><?php // drivin and rendered in javascript ?></div>
*
**************/
var objParentRateChangers = {};

(function($){

	// control class
	var Parent_Rate_Changer = function(divElement){
		var objThis = this;
		objThis.element = $(divElement);
		objThis.type = null;
		objThis.typeId = null;
		objThis.topParentId = null;
		objThis.defaultId = null;
		objThis.ignoreId = null;
		objThis.inputId = null;
		objThis.inputName = null;
		objThis.events = {};
		objThis.instanceId = null;
		objThis.modal = null;
		objThis.rates = {};
		objThis.selectedRate = {
			id: null,
			name: null
		};

		// pull in values from data attributes
		objThis.initializeValues(function(){
			objThis.createBaseHTML()
				.setupEvents()
				.search('');

			// save instance to list for access outside of this file
			objParentRateChangers[objThis.instanceId] = objThis;
		});
	}

	Parent_Rate_Changer.prototype.initializeValues = function(fnCallback){
		var objThis = this;

		objThis.instanceId = guid(); // unique instance id used for this control

		var strInstanceId = objThis.element.data('instance-id');
		if(objThis.element.attr('data-instance-id')){
			objThis.instanceId = strInstanceId;
		}

		var strHotelId = objThis.element.data('hotel-id');
		if(objThis.element.attr('data-hotel-id')){
			objThis.type = 'hotel';
			objThis.typeId = parseInt(strHotelId);
		}

		var strCourseId = objThis.element.data('course-id');
		if(objThis.element.attr('data-course-id')){
			objThis.type = 'course';
			objThis.typeId = parseInt(strCourseId);
		}

		var strTopParentId = objThis.element.data('top-parent-id');
		if(objThis.element.attr('data-top-parent-id')){
			objThis.topParentId = parseInt(strTopParentId);
		}

		var strDefaultId = objThis.element.data('default-id');
		if(objThis.element.attr('data-default-id')){
			objThis.defaultId = parseInt(strDefaultId);
		}

		var strIgnoreId = objThis.element.data('ignore-id');
		if(objThis.element.attr('data-ignore-id')){
			objThis.ignoreId = parseInt(strIgnoreId);
		}

		objThis.inputId = 'prc-value-input-' + objThis.instanceId;
		var strInputId = objThis.element.data('input-id');
		if(objThis.element.attr('data-input-id')){
			objThis.inputId = strInputId;
		}

		var strInputName = objThis.element.data('input-name');
		if(objThis.element.attr('data-input-name')){
			objThis.inputName = strInputName;
		}

		var cbEventChanged = objThis.element.data('event-change');
		if(objThis.element.attr('data-event-change')){
			objThis.events['change'] = cbEventChanged;
		}

		if(objThis.defaultId != null){
			// get default values if one is passed in
			objThis.getRateById(objThis.defaultId, fnCallback);
			return objThis;
		}

		// continue without loading a default
		if(typeof(fnCallback) == 'function'){
			fnCallback();
		}

		return objThis;
	}

	Parent_Rate_Changer.prototype.createBaseHTML = function(){
		var objThis = this;

		var strInputName = '';
		if(objThis.inputName != null){
			strInputName = 'name="' + objThis.inputName + '"';
		}

		var strSelectedName = 'No Rate Selected';
		var strSelectedId = '';

		if(objThis.selectedRate.id != null){
			strSelectedId = objThis.selectedRate.id;
			strSelectedName = objThis.selectedRate.name;
		}

		var strHTML = '<input type="hidden" id="' + objThis.inputId + '" ' + strInputName + ' class="js-prc-value-input js-prc-value-input-' + objThis.instanceId + '" value="' + strSelectedId + '" />'
					+ '<span class="js-prc-value-display">' + strSelectedName + '</span><a href="#" class="btn btn-link js-prc-open-modal">Change</a>'

		// attach to element
		objThis.element.html('').append(strHTML);

		var strModalHTML = '<div class="js-prc-holder">'
				    	+ 	'<div class="modal fade" id="prc-modal-' + objThis.instanceId + '" tabindex="-1" role="dialog" aria-labelledby="prc-modal-' + objThis.instanceId + '-label" style="z-index: 5000;">'
				        + 	'<div class="modal-dialog" role="document">'
				        + 	'<div class="modal-content">'
				        + 	'<div class="modal-header">'
				        + 	'<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>'
				        + 	'<h4 class="modal-title" id="prc-modal-' + objThis.instanceId + '-label">Select Parent Rate</h4>'
				        + 	'</div>'
				        + 	'<div class="modal-body">'
				        + 	'<input type="text" class="form-control js-prc-search" placeholder="Search" />'
						+ 	'<table class="js-prc-results table table-hover">'
				        + 	'</table>'
				        + 	'</div>'
				        + 	'<div class="modal-footer">'
				        + 	'<button type="button" class="btn btn-default" data-dismiss="modal">Close</button>'
				        + 	'</div>'
				        + 	'</div>'
				        + 	'</div>'
				   		+ 	'</div>'
						+ 	'</div>';

		// attach modal
		if(objThis.modal != null){
			// remove existing modal
			objThis.modal.parent().remove();
		}

		$('body').append(strModalHTML);
		objThis.modal = $('.js-prc-holder #prc-modal-' + objThis.instanceId);

		objThis.modal.on('shown.bs.modal', function(event){
			// move keyboard focus to search input
			$(this).find('.js-prc-search').focus();

			return true;
		});

		return objThis;
	}

	Parent_Rate_Changer.prototype.setupEvents = function(){
		var objThis = this;

		var jobjModal = $('#prc-modal-' + objThis.instanceId);

		objThis.element.on('click', '.js-prc-open-modal', function(event){
			event.preventDefault();

			objThis.showModal();
		});

		jobjModal.find('.js-prc-search').on('keyup', function(event){
			objThis.search($(this).val());
		});

		jobjModal.find('.js-prc-results').on('click', '.js-prc-view-children', function(event){
			event.preventDefault();

			var intRateId = $(this).data('rate-id');

			if(typeof(intRateId) == 'undefined' || intRateId == null){
				// no rate id found
				return false;
			}

			objThis.getChildRates(intRateId);
			return false;
		});

		jobjModal.find('.js-prc-results').on('click', '.js-prc-select-rate', function(event){
			var intRateId = $(this).data('rate-id');

			objThis.selectedRate.id = intRateId;

			if(intRateId != null){
				objThis.selectedRate.name = objThis.rates[intRateId].name;
			}else{
				objThis.selectedRate.name = null;
			}

			objThis.setSelectedRate().hideModal();
		});

		return objThis;
	}

	Parent_Rate_Changer.prototype.search = function(strSearch){
		var objThis = this;

		var strSearchType = 'base';
		if(strSearch.length > 0){
			// search all rates based on strSearch
			strSearchType = 'search';
		}

		$.getJSON('/admin/rates/' + objThis.type + '/' + objThis.typeId + '/search/' + encodeURIComponent(strSearch), {}, function(objReturn){
			if(objReturn.error === true){
				objThis.modal.find('.js-prc-results').html('<tr><td>No Rates Found.</td></tr>');
				objThis.rates = null;

				// error occured
				console.log('Unknown Error Occured: ' + objReturn.messages);
				alert('Unknown Error Occured: ' + objReturn.messages.toString());
				return false;
			}

			// success
			$.extend(objThis.rates, objReturn.rates);

			objThis.createTableRows(objReturn.rates);

			return true;
		});

		return objThis;
	}

	Parent_Rate_Changer.prototype.createTableRows = function(objRates){
		var objThis = this;
		var jobjTable = objThis.modal.find('.js-prc-results');

		if(isEmptyObject(objRates)){
			// no rates found
			jobjTable.html('<tr><td>No Rates Found.</td></tr>');
			return objThis;
		}

		var strHTML = '';

		// setup null row
		strHTML += objThis.createTableRow({
			id: null,
			name: 'No Parent Rate',
			has_children: false
		});

		for(var intRateId in objRates){
			strHTML += objThis.createTableRow(objRates[intRateId]);
		}


		jobjTable.html(strHTML);

		return objThis;
	}

	Parent_Rate_Changer.prototype.createChildRows = function(intParentId, objRates){
		var objThis = this;

		var jobjChildRowHolder = objThis.modal.find('.child-rates[data-parent-id=' + intParentId + ']');

		if(isEmptyObject(objRates)){
			// no rates found
			jobjChildRowHolder.html('<td colspan="4"><table class="child-table table table-hover"><tr><td>No Child Rates Found.</td></tr></table></td>');
			return objThis;
		}


		var strHTML = '<td colspan="4"><table class="child-table table table-hover">';

		for(var intRateId in objRates){
			strHTML += objThis.createTableRow(objRates[intRateId]);
		}

		strHTML += '</table></td>';


		jobjChildRowHolder.html(strHTML).slideDown();
	}

	Parent_Rate_Changer.prototype.createTableRow = function(objRate){
		var objThis = this;

		var strId = '';
		if(objRate.id != null){
			strId = objRate.id;
		}

		if(objRate.name == null || objRate.name.length == 0){
			// default name if there isn't one set
			objRate.name = '#' + objRate.id + ' Rate';
		}

		strHTML = '<tr class="main-rate" data-rate-id="' + objRate.id + '" data-parent-id="">'
				+ '<td>' + strId + '</td>'
				+ '<td>' + objRate.name + '</td>'


        // Prevent parents from being nested under the child records
        if (objThis.ignoreId == null || objThis.ignoreId != objRate.id) {

            // Display the child button
            if(objRate.has_children == true){
                strHTML += '<td><button class="btn btn-link js-prc-view-children" data-rate-id="' + objRate.id + '">View Children</button></td>';
            }else{
                strHTML += '<td>&nbsp;</td>';
            }

            // Display the select button
            strHTML += '<td><button class="btn btn-primary js-prc-select-rate" data-rate-id="' + objRate.id + '">select</button></td>';

        }else{
            strHTML += '<td>&nbsp;</td>';
			strHTML += '<td>&nbsp;</td>';
		}

		strHTML += '</tr>'
				 + '<tr class="child-rates" data-parent-id="' + objRate.id + '" style="display: none;"></tr>';

		return strHTML;
	}

	Parent_Rate_Changer.prototype.getChildRates = function(intRateId){
		var objThis = this;

		// get child rates
		$.getJSON('/admin/rates/' + objThis.type + '/' + objThis.typeId + '/children/' + intRateId, {}, function(objReturn){
			if(objReturn.error === true){
				objThis.modal.find('.js-prc-results').html('<td colspan="4"><table class="child-table table table-hover"><tr><td>No Rates Found.</td></tr></table></td>');
				objThis.rates = null;

				// error occured
				console.log('Unknown Error Occured: ' + objReturn.messages);
				alert('Unknown Error Occured: ' + objReturn.messages.toString());
				return false;
			}

			// success
			$.extend(objThis.rates, objReturn.rates);
		
			// create child rows
			objThis.createChildRows(intRateId, objReturn.rates);

			return true;
		});

		return objThis;
	}

	Parent_Rate_Changer.prototype.setSelectedRate = function(){
		var objThis = this;

		var strSelectedName = 'No Rate Selected';
		var strSelectedId = '';

		if(objThis.selectedRate.id != null){
			strSelectedId = objThis.selectedRate.id;
			strSelectedName = objThis.selectedRate.name;
		}

		// save rate id to input
		objThis.element.find('.js-prc-value-input-' + objThis.instanceId).val(strSelectedId);

		// change label
		objThis.element.find('.js-prc-value-display').html(strSelectedName);

		if(typeof(objThis.events['change']) == 'function'){
			objThis.events['change'].call();
		}else{
			var fnCallback = eval(objThis.events['change']);

			if(typeof(objThis.events['change']) == 'function'){
				objThis.events['change'].call();
			}
		}

		return objThis;
	}

	Parent_Rate_Changer.prototype.getSelectedRate = function(){
		var objThis = this;

		return objThis.selectedRate;
	}

	Parent_Rate_Changer.prototype.showModal = function(){
		var objThis = this;

		objThis.modal.modal('show');

		return objThis;
	}

	Parent_Rate_Changer.prototype.hideModal = function(){
		var objThis = this;

		objThis.modal.modal('hide');

		return objThis;
	}

	Parent_Rate_Changer.prototype.getRateById = function(intRateId, fnCallback){
		var objThis = this;

		// make ajax call to get default rate information
		$.getJSON('/admin/rates/' + objThis.type + '/' + objThis.typeId + '/getrate/' + intRateId, {}, function(objReturn){
			if(objReturn.error === true){
				// error occured
				console.log('Unknown Error Occured: ' + objReturn.messages);
				alert('Unknown Error Occured: ' + objReturn.messages.toString());
				return false;
			}

			// success
			objThis.rates[objReturn.rate.id] = objReturn.rate;
			
			objThis.selectedRate.id = objReturn.rate.id;

			if(objReturn.rate.name != null && objReturn.rate.name.length > 0){
				objThis.selectedRate.name = objReturn.rate.name;
			}else{
				objThis.selectedRate.name = '#' + objReturn.rate.id + ' Rate';
			}

			if(typeof(fnCallback) == 'function'){
				fnCallback();
				return objThis;
			}
		});

		return objThis;
	}

	// setup controls
	$(document).ready(function(){
		$('.js-ctrl-parent-rate-changer').each(function(){
			var objThis = this;

			var objParentRateChanger = new Parent_Rate_Changer(objThis);
		});
	});

})(jQuery)