<?php

class SEO_Scorer {

    // Controls if it runs on object creation
    public $blnStart;

    // Scores
    public $intScore;
    public $arrScoreMessages;
    public $arrScore;

    // Values we check
    public $strKeyword;
    public $arrKeywords; // A list of keywords in use already
    public $strMetaTitle;
    public $strMetaDescription;
    public $strMetaRobots;
    public $strURL;
    public $strContent;

    public function __construct($arrParams) {

        $this->blnStart = isset($arrParams['start']) ? $arrParams['start'] : true;

        $this->strKeyword = isset($arrParams['keyword']) ? strtolower($arrParams['keyword']) : '';
        $this->arrKeywords = isset($arrParams['keyword_array']) ? array_filter($arrParams['keyword_array']) : array();
        $this->strMetaTitle = isset($arrParams['meta_title']) ? $arrParams['meta_title'] : '';
        $this->strMetaDescription = isset($arrParams['meta_description']) ? $arrParams['meta_description'] : '';
        $this->strMetaRobots = isset($arrParams['meta_robots']) ? $arrParams['meta_robots'] : '';
        $this->strContent = isset($arrParams['content']) ? $arrParams['content'] : '';
        $this->strURL = isset($arrParams['url']) ? $arrParams['url'] : '';

        // Set the score
        $this->intScore = 0;
        $this->arrScoreMessages = array();
        $this->arrScore = array(
            'score' => $this->intScore,
            'messages' => $this->arrScoreMessages
        );

        // Score the content
        if ((boolean) $this->blnStart == true) {
            $this->scoreSEO();
        }
    }

    /**
     * Wrapper functions that scores all the SEO
     * Gets run on init to create the score
     */
    public function scoreSEO() {
        $arrScoreReturns = array();

        // Exit if the Keyword is blank
        if (empty($this->strKeyword)) {
            $this->arrScoreMessages[] = array(
                'type' => 'low',
                'message' => 'A focus keyword was not set, a SEO score can not be generated.'
            );
            $this->arrScore['messages'] = $this->arrScoreMessages;
            return;
        }

        // Exit if the Robots are set to noindex
        if (stripos($this->strMetaRobots, 'noindex') !== false) {
            $this->arrScoreMessages[] = array(
                'type' => 'low',
                'message' => 'The meta robots are set to not index this page, a SEO score can not be generated.'
            );
            $this->arrScore['messages'] = $this->arrScoreMessages;
            return;
        }

        // Score the content
        $arrScoreReturns[] = $this->scoreKeyword($this->strKeyword);
        //$arrScoreReturns[] = $this->scoreKeywordArray($this->arrKeywords, $this->strKeyword);
        $arrScoreReturns[] = $this->scoreURL($this->strURL, $this->strKeyword);
        $arrScoreReturns[] = $this->scoreTitleLength($this->strMetaTitle);
        $arrScoreReturns[] = $this->scoreTitleKeywordPosition($this->strMetaTitle, $this->strKeyword);
        $arrScoreReturns[] = $this->scoreDescriptionLength($this->strMetaDescription);
        $arrScoreReturns[] = $this->scoreDescriptionKeywordPosition($this->strMetaDescription, $this->strKeyword);
        $arrScoreReturns[] = $this->scoreContentWordCount($this->strContent);
        $arrScoreReturns[] = $this->scoreContentReadingLevel($this->strContent);
        $arrScoreReturns[] = $this->scoreContentKeywordDensity($this->strContent, $this->strKeyword);

        // Create the parsed HTML
        $resHTML = new DOMDocument();
        @$resHTML->loadHTML($this->strContent); // Suppress errors if the HTML is invalid

        // These methods use DOM parsing to find their respective tags
        $arrScoreReturns[] = $this->scoreHTMLFirstParagraph($resHTML, $this->strKeyword);
        $arrScoreReturns[] = $this->scoreHTMLHeadings($resHTML, $this->strKeyword);
        $arrScoreReturns[] = $this->scoreHTMLImages($resHTML, $this->strKeyword);
        $arrScoreReturns[] = $this->scoreHTMLLinks($resHTML, $this->strKeyword);

        // Remove empty values and add it to the messages
        $this->arrScoreMessages = array_filter($arrScoreReturns);

        // Make sure the score can't be below 0 or above 100
        if ($this->intScore < 0) {
            $this->intScore = 0;
        } else if ($this->intScore > 100) {
            $this->intScore = 100;
        }

        // Set the class value
        $this->arrScore = array(
            'score' => $this->intScore,
            'messages' => $this->arrScoreMessages
        );

    }

    /**
     * Getters
     *
     */
    public function getScore() { return $this->arrScore; }
    public function getScoreNumber() { return $this->intScore; }
    public function getScoreMessages() { return $this->arrScoreMessages; }

    /**
     * Checks the Keword for stopwords
     *
     */
    public function scoreKeyword($strKeyword) {
        $arrScoreMessages = array();

        // Exit case
        if (empty($strKeyword)) {
            $arrScoreMessages = array(
                'type' => 'low',
                'message' => 'The focus keyword is empty.'
            );
            return $arrScoreMessages;
        }

        // Check for stop words
        if (preg_match('/\b(a|in|an|on|for|the|and)\b/i', $strKeyword)) {
            $arrScoreMessages = array(
                'type' => 'low',
                'message' => 'The focus keyword for this page contains one or more stop words (a, in, an, on, for, the, and) consider removing them.'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Checks an array of keywords to see if it's already been used
     *
     */
    public function scoreKeywordArray($arrKeywords, $strKeyword) {
        $arrScoreMessages = array();

        // Exit case
        if (empty($arrKeywords) || !is_array($arrKeywords)) {
            $arrScoreMessages = array(
                'type' => 'low',
                'message' => 'There are no other pages with focus keywords to compare against.'
            );
            return $arrScoreMessages;
        }

        // Convert to lower case and count the number of times a keyword has been used
        $arrKeywordValues = array_count_values(array_map('strtolower', $arrKeywords));
        $intKeywordCount = $arrKeywordValues[$strKeyword];

        if ($intKeywordCount <= 1) {
            // Not used before
            $this->intScore += 9;
            $arrScoreMessages = array(
                'type' => 'high',
                'message' => 'You have not used the focus keyword <strong>'.$strKeyword.'</strong> before, very good.'
            );
        } else if ($intKeywordCount == 2) {
            // Used once before
            $this->intScore += 6;
            $arrScoreMessages = array(
                'type' => 'medium',
                'message' => 'You have used the focus keyword <strong>'.$strKeyword.'</strong> before, It is recommended to have a unique focus keyword for every url.'
            );
        } else if ($intKeywordCount > 2) {
            // Used more then once
            $this->intScore += 1;
            $arrScoreMessages = array(
                'type' => 'low',
                'message' => 'You have used this focus keyword <strong>'.$strKeyword.'</strong>, ('.$intKeywordCount.') times before, It is recommended to have a unique focus keyword for every url.'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Scores the URL, by checking for the keyword
     *
     */
    public function scoreURL($strURL, $strKeyword) {
        $arrScoreMessages = array();
        $strURL = str_replace(array('-', '_', '/'), ' ', $strURL);
        $intKeywordPosition = stripos($strURL, $strKeyword);

        // Exit case
        if ($intKeywordPosition === false) {
            $this->intScore += 6;
                $arrScoreMessages = array(
                'message' => 'The focus keyword <strong>'.$strKeyword.'</strong>, does not appear in the URL for this page.',
                'type' => 'medium'
            );
            return $arrScoreMessages;
        }

        $this->intScore += 9;
        $arrScoreMessages = array(
            'message' => 'The focus keyword <strong>'.$strKeyword.'</strong> appears in the URL for this page.',
            'type' => 'high'
        );

        return $arrScoreMessages;
    }

    /**
     * Scores the length of the meta title
     *
     */
    public function scoreTitleLength($strMetaTitle) {
        $arrScoreMessages = array();
        $intTitleLength = strlen($strMetaTitle);

        // Exit case
        if ($intTitleLength == 0) {
            $this->intScore += 1;
            $arrScoreMessages = array(
                'message' => 'You do not have a page meta title, please create a page title.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }

        if ($intTitleLength < 40){
            $this->intScore += 6;
            $arrScoreMessages = array(
                'message' => 'The page meta title contains <strong>'.$intTitleLength.'</strong> characters, which is less than the recommended minimum of 40 characters.',
                'type' => 'medium'
            );

        } else if ($intTitleLength > 70) {
            $this->intScore += 6;
            $arrScoreMessages = array(
                'message' => 'The page meta title contains <strong>'.$intTitleLength.'</strong> characters, which is more than the viewable limit of 70 characters; some words will not be visible.',
                'type' => 'medium'
            );

        } else {
            $this->intScore += 9;
            $arrScoreMessages = array(
                'message' => 'The page meta title is more than <strong>40</strong> characters and less than the recommended <strong>70</strong> character limit.',
                'type' => 'high'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Scores the position of the keyword in the meta title
     *
     */
    public function scoreTitleKeywordPosition($strMetaTitle, $strKeyword) {
        $arrScoreMessages = array();
        $intKeywordPosition = stripos($strMetaTitle, $strKeyword);

        // Exit case
        if ($intKeywordPosition === false) {
            $this->intScore += 2;
            $arrScoreMessages = array(
                'message' => 'The focus keyword <strong>'.$strKeyword.'</strong>, does not appear in the page meta title.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }

        if ($intKeywordPosition == 0) {
            $this->intScore += 9;
            $arrScoreMessages = array(
                'message' => 'The page meta title contains your focus keyword, and the focus keyword is at the beginning which is considered to improve rankings.',
                'type' => 'high'
            );

        } else {
            $this->intScore += 6;
            $arrScoreMessages = array(
                'message' => 'The page meta title contains your focus keyword, but it does not appear at the beginning; try and move it to the beginning.',
                'type' => 'medium'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Scores the meta description length
     *
     */
    public function scoreDescriptionLength($strMetaDescription) {
        $arrScoreMessages = array();
        $intDescriptionLength = strlen($strMetaDescription);

        // Exit case
        if ($intDescriptionLength == 0) {
            $this->intScore += 1;
            $arrScoreMessages = array(
                'message' => 'No meta description has been specified, search engines will display copy from the page instead.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }

        if ($intDescriptionLength > 153){
            $this->intScore += 6;
            $arrScoreMessages = array(
                'message' => 'Your meta description is over <strong>153</strong> characters, reducing it will ensure the entire description is visible.',
                'type' => 'low'
            );

        } else if ($intDescriptionLength == 153) {
            $this->intScore += 7;
            $arrScoreMessages = array(
                'message' => 'How does your description compare to the competition? Could it be made more appealing?',
                'type' => 'high'
            );

        } else if ($intDescriptionLength >= 120) {
            $this->intScore += 7;
            $arrScoreMessages = array(
                'message' => 'The meta description is over the <strong>120</strong> characters minimum.',
                'type' => 'high'
            );

        } else {
            $this->intScore += 6;
            $arrScoreMessages = array(
                'message' => 'The meta description is under <strong>120</strong> characters.',
                'type' => 'low'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Scores if the keyword exists in the meta description
     *
     */
    public function scoreDescriptionKeywordPosition($strMetaDescription, $strKeyword) {
        $arrScoreMessages = array();
        $intDescriptionPosition = stripos($strMetaDescription, $strKeyword);

        // Exit case
        if ($intDescriptionPosition === false) {
            $arrScoreMessages = array(
                'message' => 'The focus keyword <strong>'.$strKeyword.'</strong>, does not appear in the meta description.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }

        $this->intScore += 8;
        $arrScoreMessages = array(
            'message' => 'The meta description contains your focus keyword.',
            'type' => 'high'
        );

        return $arrScoreMessages;
    }

    /**
     * Scores the content word count
     *
     */
    public function scoreContentWordCount($strContent) {
        $arrScoreMessages = array();
        $intContentWordCount = str_word_count(strip_tags($strContent));

        // Exit case
        if (empty($intContentWordCount)) {
            $arrScoreMessages = array(
                'message' => 'The content is empty, a word count cannot be generated.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }

        if($intContentWordCount >= 300) {
            $this->intScore += 9;
            $arrScoreMessages = array(
                'message' => 'There are <strong>'.$intContentWordCount.'</strong> words contained in the body copy, this is following the <strong>300</strong> word recommended minimum.',
                'type' => 'high'
            );

        } else if ($intContentWordCount >= 250) {
            $this->intScore += 7;
            $arrScoreMessages = array(
                'message' => 'There are <strong>'.$intContentWordCount.'</strong> words contained in the body copy, this is slightly below the <strong>300</strong> word recommended minimum, add a bit more copy.',
                'type' => 'medium'
            );

        } else if ($intContentWordCount >= 200) {
            $intContentWordCount += 5;
            $arrScoreMessages = array(
                'message' => 'There are <strong>'.$intContentWordCount.'</strong> words contained in the body copy, this is below the <strong>300</strong> word recommended minimum. Add more useful content on this topic for readers.',
                'type' => 'medium'
            );

        } else if ($intContentWordCount >= 100) {
            $this->intScore += -10;
            $arrScoreMessages = array(
                'message' => 'There are  <strong>'.$intContentWordCount.'</strong> words contained in the body copy. This is far too low and should be increased to be at least <strong>300</strong> words.',
                'type' => 'low'
            );

        } else {
            $this->intScore += -20;
            $arrScoreMessages = array(
                'message' => 'Warning, you only have <strong>'.$intContentWordCount.'</strong> words contained in the body copy. This is far below the <strong>300</strong> word minimum.',
                'type' => 'low'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Scores the density of the keyword in the content
     *
     */
    public function scoreContentKeywordDensity($strContent, $strKeyword) {
        $arrScoreMessages = array();
        $intContentWordCount = str_word_count(strip_tags($strContent));

        // Exit case
        if (empty($intContentWordCount) || empty($strKeyword)) {
            $arrScoreMessages = array(
                'message' => 'The content is empty, a keyword density cannot be generated.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }

        $strPattern = '/\b'.$strKeyword.'\b/i'; // caseless whole word pattern
        $intKeywordCount = preg_match_all($strPattern, $strContent, $arrMatches);
        $intKeywordDensity = ($intKeywordCount / $intContentWordCount) * 100;

        // Get Keyword Density Score
        if ($intKeywordDensity > 4.5)  {
            $this->intScore += -50;
            $arrScoreMessages = array(
                'message' => 'The keyword density is <strong>'.round($intKeywordDensity, 2).'</strong> which is over the advised 4.5% maximum, the keyword was found <strong>'.$intKeywordCount.'</strong> times.',
                'type' => 'low'
            );

        } else if ($intKeywordDensity >= 2)  {
            $this->intScore += 9;
            $arrScoreMessages = array(
                'message' => 'The keyword density is <strong>'.round($intKeywordDensity, 2).'</strong> , which is great, the keyword was found <strong>'.$intKeywordCount.'</strong> times.',
                'type' => 'high'
            );

        } else if ($intKeywordDensity >= 1){
            $this->intScore += 4;
            $arrScoreMessages = array(
                'message' => 'The keyword density is <strong>'.round($intKeywordDensity, 2).'</strong> , which is  low, the keyword was found <strong>'.$intKeywordCount.'</strong> times.',
                'type' => 'medium'
            );

        } else {
            $this->intScore += 0;
            $arrScoreMessages = array(
                'message' => 'The keyword density is <strong>'.round($intKeywordDensity, 2).'</strong> which is low, this will lower your chances of being found in the search engines.',
                'type' => 'low'
            );

        }

        return $arrScoreMessages;
    }

    /**
     * Scores the first paragraph in the content
     * requires the content to be parsed by DOMDocument
     */
    public function scoreHTMLFirstParagraph($resHTML, $strKeyword) {
        $arrScoreMessages = array();
        $objParagraphs = $resHTML->getElementsByTagName('p');

        // Exit case
        if ($objParagraphs->length == 0) {
            $arrScoreMessages = array(
                'message' => 'You do not have any paragraph tags in your content, double check your HTML to make sure it is valid.',
                'type' => 'medium'
            );
        }

        foreach ($objParagraphs as $objParagraph) {
            $strParagraph = $objParagraph->nodeValue;
            $intKeywordPosition = stripos($strParagraph, $strKeyword);

            if ($intKeywordPosition !== false) {
                $this->intScore += 5;
                $arrScoreMessages = array(
                    'message' => 'Your focus keyword occurs in your first paragraph.',
                    'type' => 'high'
                );
            } else {
                $this->intScore += 3;
                $arrScoreMessages = array(
                    'message' => 'Your focus keyword <strong>'. $strKeyword .'</strong>, does not occur in your first paragraph, it would be a good idea to include it at least once.',
                    'type' => 'low'
                );
            }

            break; // Exit after the first paragraph
        }

        return $arrScoreMessages;
    }

    /**
     * Scores the headings in the content
     * requires the content to be parsed by DOMDocument
     */
    public function scoreHTMLHeadings($resHTML, $strContent) {
        $arrScoreMessages = array();
        $arrHeadings = array();

        for ($i=1; $i <= 6; $i++) {
            $strTag = "h$i";
            $objHeadings = $resHTML->getElementsByTagName($strTag);

            if ($objHeadings->length > 0) {
                $arrHeadings[$strTag] = $objHeadings;
            }
        }

        // Exit case
        if (empty($arrHeadings)) {
            $arrScoreMessages = array(
                'message' => 'No subheading tags (like an H2) appear in your content.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }

        // Set a variable for if we find a heading
        $blnFound = false;

        // Loop through the heading types
        foreach ($arrHeadings as $objHeading) {

            // Loop through the items in that obj
            foreach ($objHeading as $objHeading) {
                $strHeading = $objHeading->nodeValue;
                if (stripos($strHeading, $this->strKeyword) !== false) {
                    $blnFound = true;
                    $this->intScore += 8;
                    $arrScoreMessages = array(
                        'message' => 'The focus keyword appears in the subheadings (like an H2). While not a major ranking factor, this is beneficial.',
                        'type' => 'high'
                    );
                }

                if ($blnFound) break;
            }

        }

        if (!$blnFound){
            $this->intScore += 3;
            $arrScoreMessages = array(
                'message' => 'You have not used your focus keyword <strong>'.$this->strKeyword.'</strong>, in any subheading (such as an H2) in your content.',
                'type' => 'low'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Scores the images in the content
     * requires the content to be parsed by DOMDocument
     */
    public function scoreHTMLImages($resHTML, $strKeyword) {
        $arrScoreMessages = array();
        $objImages = $resHTML->getElementsByTagName('img');

        // Exit case
        if ($objImages->length == 0) {
            $arrScoreMessages = array(
                'message' => 'You currently do not have any images, images with alt tags using your focus keyword can help your ranking.',
                'type' => 'medium'
            );
            return $arrScoreMessages;
        }

        // Set a variable for if we find an image with the alt keyword
        $blnFound = false;

        // Loop through the images
        foreach ($objImages as $image) {
            $strAltText = $image->getAttribute('alt');
            if (stripos($strAltText, $strKeyword) !== false) {
                $blnFound = true;
                $this->intScore += 4;
                $arrScoreMessages = array(
                    'message' => 'The focus keyword appears in an alt tag of an image.',
                    'type' => 'high'
                );
            }

            if ($blnFound) break;
        }

        if (!$blnFound) {
            // If there are no images in the content
            $this->intScore += 1;
            $arrScoreMessages = array(
                'message' => 'You do not have any images in your content that use your focus keyword <strong>'.$strKeyword.'</strong> in their alt attribute.',
                'type' => 'medium'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Scores the links in the content
     * requires the content to be parsed by DOMDocument
     */
    public function scoreHTMLLinks($resHTML, $strKeyword) {
        $arrScoreMessages = array();
        $objLinks = $resHTML->getElementsByTagName('a');

        // Exit case
        if ($objLinks->length == 0) {
            $this->intScore += 3;
            $arrScoreMessages = array(
                'message' => 'No outbound links appear in this page, consider adding some as appropriate.',
                'type' => 'medium'
            );
            return $arrScoreMessages;
        }

        // Set a variable for if we find an external link
        $blnFound = false;

        // Loop through the links
        foreach ($objLinks as $link) {
            $strLinkText = $link->nodeValue;
            $strLinkURL = parse_url($link->getAttribute('href'));

            // Only links that go to another site
            if (isset($strLinkURL['host']) && $strLinkURL['host'] != $_SERVER['HTTP_HOST']) {

                if (stripos($strLinkText, $strKeyword) !== false) {
                    $blnFound = true;
                    $this->intScore += 5;
                    $arrScoreMessages = array(
                        'message' => 'You are using the focus keyword <strong>'.$strKeyword.'</strong> in an outbound link.',
                        'type' => 'high'
                    );
                }

            }

            if ($blnFound) break;
        }

        if (!$blnFound) {
            // If there are no images in the content
            $this->intScore += 1;
            $arrScoreMessages = array(
                'message' => 'You are not using an outbound link with the focus keyword <strong>'.$strKeyword.'</strong>, consider adding one to improve rankings.',
                'type' => 'medium'
            );
        }

        return $arrScoreMessages;
    }

    /**
     * Scores the reading level of the content text
     * relies on the TextStatistics class to run the grading method
     */
    public function scoreContentReadingLevel($strContent) {
        $arrScoreMessages = array();
        $strContent = strip_tags($strContent);

        // Exit cases
        if (!class_exists('TextStatistics')) {
            $arrScoreMessages = array(
                'message' => 'The text grading class is not included, a reading level cannot be generated.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }
        if (empty($strContent)) {
            $arrScoreMessages = array(
                'message' => 'The content is empty, a reading level cannot be generated.',
                'type' => 'low'
            );
            return $arrScoreMessages;
        }

        $objReadingLevelGrader = new TextStatistics;
        $intReadingLevel = $objReadingLevelGrader->flesch_kincaid_reading_ease($strContent);

        // Set the grades.
        if ($intReadingLevel >= 90) {
            $this->intScore += 9;
            $arrScoreMessages = array(
                'message' => 'The content is very easy to read.',
                'type' => 'high'
            );

        } elseif ($intReadingLevel >= 80) {
            $this->intScore += 9;
            $arrScoreMessages = array(
                'message' => 'The content is easy to read.',
                'type' => 'high'
            );

        } elseif ($intReadingLevel >= 70) {
            $this->intScore += 8;
            $arrScoreMessages = array(
                'message' => 'The content is fairly easy to read.',
                'type' => 'medium'
            );

        } elseif ($intReadingLevel >= 60) {
            $this->intScore += 7;
            $arrScoreMessages = array(
                'message' => 'The content can be read by most people.',
                'type' => 'medium'
            );

        } elseif ($intReadingLevel >= 50) {
            $this->intScore += 6;
            $arrScoreMessages = array(
                'message' => 'The content is fairly difficult to read, try to make shorter sentences to improve readability.',
                'type' => 'low'
            );

        } elseif ($intReadingLevel >= 30) {
            $this->intScore += 5;
            $arrScoreMessages = array(
                'message' => 'The content is difficult to read, try to make shorter sentences, using less difficult words to improve readability.',
                'type' => 'low'
            );

        } elseif ($intReadingLevel >= 0) {
            $this->intScore += 4;
            $arrScoreMessages = array(
                'message' => 'The content is very difficult to read, try to make shorter sentences and use less difficult words to improve readability.',
                'type' => 'low'
            );

        }

        return $arrScoreMessages;
    }


}
