<?php

/**
 * Menu builder for the frontend
 *
 */
class Menu extends Database {

    public $data;
    public $html;

    // Build the menu
    function __construct() {
        // Init the DB
        parent::__construct();

        // Build the menu structure
        $this->data = $this->build();
        $this->html = $this->html();
    }

    /**
     * Get a menu property
     *
     */
    public function get($key) {
        return isset($this->$key) ? $this->$key: false;
    }

    /**
     * Outputs the HTML
     *
     */
    public function output() {
        echo $this->html;
    }

    /**
     * Builds the menu based off the passed data
     *
     */
    public function build() {
        // Get the dynamic menu items
        $arrRegionItems = $this->getRegionItems();
        $arrPackageItems = $this->getPackageItems();
        $arrPageItems = $this->getPageItems();

        // Build the menu structure
        $data = array(
            array(
                'title' => 'Golf Regions',
                'subtitle' => 'Browse Golf Regions',
                'type' => 'megamenu',
                'data' => array(
                    array(
                        'title' => 'British Columbia',
                        'data' => $arrRegionItems['bc'],
                    ),
                    array(
                        'title' => 'Alberta and West',
                        'data' => $arrRegionItems['ca'],
                    ),
                    array(
                        'title' => 'USA',
                        'data' => $arrRegionItems['usa'],
                    ),
                    array(
                        'title' => 'International',
                        'data' => $arrRegionItems['int'],
                    ),
                    array(
                        'title' => 'More',
                        'data' => $arrRegionItems['other'],
                    ),
                ),
            ),
            array(
                'title' => 'Golf Vacation Packages',
                'subtitle' => 'Browse Golf Vacation Packages',
                'type' => 'megamenu',
                'data' => array(
                    array(
                        'title' => 'British Columbia',
                        'data' => $arrPackageItems['bc'],
                    ),
                    array(
                        'title' => 'Alberta and West',
                        'data' => $arrPackageItems['ca'],
                    ),
                    array(
                        'title' => 'USA',
                        'data' => $arrPackageItems['usa'],
                    ),
                    array(
                        'title' => 'International',
                        'data' => $arrPackageItems['int'],
                    ),
                    array(
                        'title' => 'More',
                        'data' => $arrPackageItems['other'],
                    ),
                    array(
                        'title' => '',
                        'data' => array(
                            array(
                                'title' => '<strong>View All Featured Packages</strong>',
                                'url' => ROOT_URL . '/featured-golf-packages/',
                            ),
                        ),
                    ),
                ),
            ),
            array(
                'title' => 'BC Golf Course Map',
                'type' => 'menu_item',
                'url' => ROOT_URL . '/bc-golf-course-map/',
            ),
            array(
                'title' => 'Golf Travel Resources',
                'type' => 'dropdown',
                'data' => $arrPageItems,
            ),
        );

        return $data;
    }

    /**
     * Adds the html structure
     *
     */
    public function html() {
        // Build the HTML
        $html = '';
        foreach ($this->data as $item) {
            switch ($item['type']) {
                case 'menu_item':
                    $html .= self::createMenuItem($item);
                    break;
                case 'dropdown':
                    $html .= self::createDropdown($item);
                    break;
                case 'megamenu':
                    $html .= self::createMegamenu($item);
                    break;
            }
        }

        return $html;
    }

    /**
     * Creates a menu item
     *
     */
    public static function createMenuItem($arrArgs = array()) {
        // Set default args
        $arrArgs = Util::wp_parse_args($arrArgs, array(
            'title' => '',
            'url' => '',
        ));

        // Exit if no URL
        if (empty($arrArgs['url'])) {
            return;
        }

        return '<li><a href="' . $arrArgs['url'] . '">' . $arrArgs['title'] . '</a></li>';
    }

    /**
     * Creates a dropdown
     *
     */
    public static function createDropdown($arrArgs = array()) {
        // Set default args
        $arrArgs = Util::wp_parse_args($arrArgs, array(
            'title' => '',
            'data' => array(),
        ));

        // Exit if no data
        if (empty($arrArgs['data'])) {
            return;
        }

        // Build the items
        $strMenuItems = '';
        foreach ($arrArgs['data'] as $arrItem) {
            $strMenuItems .= self::createMenuItem($arrItem);
        }

        return '<li class="dropdown">'
             . '<a href="#" class="dropdown-toggle" data-toggle="dropdown">' . $arrArgs['title'] . ' <span class="caret"></span></a>'
             . '<ul class="dropdown-menu" role="menu">' . $strMenuItems . '</ul>'
             . '</li>';
    }

    /**
     * Creates a megamenu
     *
     */
    public static function createMegamenu($arrArgs = array()) {
        // Set default args
        $arrArgs = Util::wp_parse_args($arrArgs, array(
            'title' => '',
            'subtitle' => '',
            'data' => array(),
        ));

        // Build the groups
        $strGroups = '';
        foreach ($arrArgs['data'] as $arrGroup) {
            // Exit if no items
            if (empty($arrGroup['data'])) {
                continue;
            }

            // Build the items
            $strMenuItems = '';
            foreach ($arrGroup['data'] as $arrItem) {
                $strMenuItems .= self::createMenuItem($arrItem);
            }

            // Add the wrapper
            $strGroups .= '<p class="column-title">' . $arrGroup['title'] . '</p>'
                        . '<ul class="list-column">' . $strMenuItems . '</ul>';
        }

        // Exit if no groups
        if (empty($strGroups)) {
            return;
        }

        return '<li class="dropdown yamm-fw">'
             . '<a href="#" class="dropdown-toggle" data-toggle="dropdown">' . $arrArgs['title'] . ' <span class="caret"></span></a>'
             . '<ul class="dropdown-menu" role="menu">'
             . '<li>'
             . '<div class="yamm-title">' . $arrArgs['subtitle'] . '</div>'
             . '<div class="yamm-content">' . $strGroups . '</div>'
             . '</li>'
             . '</ul>'
             . '</li>';
    }

    /**
     * Gets the regions menu data
     *
     */
    public function getRegionItems() {
        // Get the settings object
        global $objSettings;

        // Set variables and get items
        $arrItems = array();
        $arrRegionData = array(
            'bc' => $objSettings->getRegionsForNav(array('state' => 2)), // BC
            'ca' => $objSettings->getRegionsForNav(array('country' => 261)), // CA
            'usa' => $objSettings->getRegionsForNav(array('country' => 148)), // USA
            'int' => $objSettings->getRegionsForNav(array('international' => true)), // INTERNATIONAL
            'other' => $objSettings->getRegionsForNav(array('other' => true)), // OTHER
        );

        // Format the urls
        foreach ($arrRegionData as $key => $arrData) {
            // Set default
            $arrItems[$key] = array();

            foreach ($arrData as $arrRegion) {
                // If on the Canadian region, skip BC as it's already included separately
                if ($key == 'ca' && (isset($arrRegion['state_id']) && $arrRegion['state_id'] == 2)) {
                    continue;
                }

                $arrItems[$key][] = array(
                    'title' => $arrRegion['region_name'],
                    'url' => Util::getPermalink('region', $arrRegion['slug']),
                );
            }
        }

        return $arrItems;
    }

    /**
     * Gets the package menu data
     *
     */
    public function getPackageItems() {
        // Get the settings object
        global $objSettings;

        // Set variables and get items
        $arrItems = array();
        $arrRegionData = array(
            'bc' => $objSettings->getRegionsForNav(array('state' => 2, 'packages' => true)), // BC
            'ca' => $objSettings->getRegionsForNav(array('country' => 261, 'packages' => true)), // CA
            'usa' => $objSettings->getRegionsForNav(array('country' => 148, 'packages' => true)), // USA
            'int' => $objSettings->getRegionsForNav(array('international' => true, 'packages' => true)), // INTERNATIONAL
            'other' => $objSettings->getRegionsForNav(array('other' => true, 'packages' => true)), // OTHER
        );

        // Format the package urls
        foreach ($arrRegionData as $key => $arrData) {
            // Set default
            $arrItems[$key] = array();

            foreach ($arrData as $arrRegion) {
                // If on the canadian region, skip BC as it's already included
                if ($key == 'ca' && (isset($arrRegion['state_id']) && $arrRegion['state_id'] == 2)) {
                    continue;
                }

                $arrItems[$key][] = array(
                    'title' => $arrRegion['region_name'],
                    'url' => Util::getPermalink('region_package', $arrRegion['slug']),
                );
            }
        }

        return $arrItems;
    }

    /**
     * Gets the page menu data
     *
     */
    public function getPageItems() {
        // Get the settings object
        global $objSettings;

        // Set variables and get data
        $arrItems = array();
        $arrPageData = $objSettings->getPagesForNav(7);

        // Add the golf blog to the start of the page data
        $arrItems[] = array(
            'title' => 'Golf Blog',
            'url' => 'https://www.golfblog.bcgolfguide.com',
        );

        // Format the pages
        foreach ($arrPageData as $arrPage) {
            $arrItems[] = array(
                'title' => $arrPage['title'],
                'url' => Util::getPermalink('page', $arrPage['slug']),
            );
        }

        return $arrItems;
    }

}
